/*
 *  SndMail by Davide Libenzi ( SMTP Internet mail delivery agent )
 *  Copyright (C) 2000, 2001  Davide Libenzi
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Davide Libenzi <davidel@xmailserver.org>
 *
 */


#include <winsock2.h>
#include <mswsock.h>
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include "sockio.h"



#define MIN_STEP_TIMEOUT            80
#define MIN_RTX_BLOCK_SIZE          512
#define SHUTDOWN_RECV_TIMEOUT       4






static int      CallIdleProc(void);
static int      SvrWaitAsyncResponse(SOCKET SockFD, WSAEVENT hRtxEvent, WSAOVERLAPPED * pOVLP,
                        int iTimeout = 0);
static int      SvrSendLL(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags);
static int      SvrRecvLL(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags);





static DWORD    dwStepTimeout = 200;
static int      iRTXBlockSize = 4 * 1024;
static CRITICAL_SECTION csIdle;
static int      (*pSockIdleProc) (void *) = NULL;
static void    *pSockIdleData = NULL;
static int      (*pLocalSockIdleProc) (void *) = NULL;
static void    *pLocalSockIdleData = NULL;








//
// Funzione SVRINITSOCKETS
//
// @func BOOL | SvrInitSockets | DESCRIZIONE
//
// @parm char | *pszSockDesc | DESCRIZIONE
// @parm *pIdleProc)(void | *) | DESCRIZIONE
// @parm void | *pIdleData | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
BOOL            SvrInitSockets(char *pszSockDesc, int (*pIdleProc) (void *), void *pIdleData)
{

    WSADATA         wsd;

    ZeroMemory(&wsd, sizeof(wsd));
    if (WSAStartup(SOCK_VERSION_REQUESTED, &wsd))
        return (FALSE);

    InitializeCriticalSection(&csIdle);

    pSockIdleProc = pIdleProc;
    pSockIdleData = pIdleData;

    if (pszSockDesc != NULL)
        strcpy(pszSockDesc, wsd.szDescription);

    return (TRUE);

}






//
// Funzione SVRCLEANUPSOCKETS
//
// @func void | SvrCleanupSockets | DESCRIZIONE
//
// @parm  | void | DESCRIZIONE
//
void            SvrCleanupSockets(void)
{

    WSACleanup();
    DeleteCriticalSection(&csIdle);

}






//
// Funzione SVRSETSTEPTIMEOUT
//
// @func void | SvrSetStepTimeout | DESCRIZIONE
//
// @parm DWORD | dwTimeout | DESCRIZIONE
//
void            SvrSetStepTimeout(DWORD dwTimeout)
{

    dwStepTimeout = max(dwTimeout, MIN_STEP_TIMEOUT);

}






//
// Funzione SVRSETRTXBLOCKSIZE
//
// @func void | SvrSetRTXBlockSize | DESCRIZIONE
//
// @parm int | iRTXSize | DESCRIZIONE
//
void            SvrSetRTXBlockSize(int iRTXSize)
{

    iRTXBlockSize = max(iRTXSize, MIN_RTX_BLOCK_SIZE);

}





//
// Funzione SVRSETBLOCKING
//
// @func int | SvrSetBlocking | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm int | on | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
int             SvrSetBlocking(SOCKET SockFD, int on)
{

    u_long          IoctlLong = (on) ? 0 : 1;

    return (ioctlsocket(SockFD, FIONBIO, &IoctlLong));

}




//
// Funzione CALLIDLEPROC
//
// @func static int | CallIdleProc | DESCRIZIONE
//
// @parm  | void | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
static int      CallIdleProc(void)
{

    int             iRetValue = 0;

    EnterCriticalSection(&csIdle);

    if (pSockIdleProc != NULL)
        iRetValue = pSockIdleProc(pSockIdleData);

    if ((iRetValue == 0) && (pLocalSockIdleProc != NULL))
        iRetValue = pLocalSockIdleProc(pLocalSockIdleData);

    LeaveCriticalSection(&csIdle);

    return (iRetValue);

}




//
// Funzione SVRSETLOCALCALLBACK
//
// @func BOOL | SvrSetLocalCallback | DESCRIZIONE
//
// @parm int(*pLocalIdleProc)(void | *) | DESCRIZIONE
// @parm void | *pLocalIdleData | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
BOOL            SvrSetLocalCallback(int (*pLocalIdleProc) (void *), void *pLocalIdleData)
{

    EnterCriticalSection(&csIdle);

    pLocalSockIdleProc = pLocalIdleProc;
    pLocalSockIdleData = pLocalIdleData;

    LeaveCriticalSection(&csIdle);

    return (TRUE);

}




//
// Funzione SVRCONNECTTIMEOUT
//
// @func int | SvrConnectTimeout | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm const struct sockaddr * | pSockName | DESCRIZIONE
// @parm int | iNameLen | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
int             SvrConnectTimeout(SOCKET SockFD, const struct sockaddr * pSockName, int iNameLen,
                        int iTimeout)
{

    int             iConnectResult,
                    iConnectError;
    WSAEVENT        hConnectEvent;

    if ((hConnectEvent = WSACreateEvent()) == NULL)
        return (-1);

    WSAEventSelect(SockFD, hConnectEvent, FD_CONNECT);

    iConnectResult = WSAConnect(SockFD, pSockName, iNameLen, NULL, NULL, NULL, NULL);

    iConnectError = WSAGetLastError();

    if ((iConnectResult != 0) && (iConnectError == WSAEWOULDBLOCK))
    {
        DWORD           dwWaitResult,
                        dwEntryCount = GetTickCount();

        while ((dwWaitResult = WSAWaitForMultipleEvents(1, &hConnectEvent, FALSE,
                                dwStepTimeout, TRUE)) != WSA_WAIT_EVENT_0)
        {
            if (dwWaitResult == WSA_WAIT_FAILED)
                break;

            if ((iTimeout > 0) &&
                    ((iTimeout * 1000) < (int) (GetTickCount() - dwEntryCount)))
                break;

            if (CallIdleProc() < 0)
                break;
        }
        iConnectResult = (dwWaitResult == WSA_WAIT_EVENT_0) ? 0 : -1;
    }

    WSAEventSelect(SockFD, hConnectEvent, 0);
    WSACloseEvent(hConnectEvent);

    return (iConnectResult);

}




//
// Funzione SVRACCEPTTIMEOUT
//
// @func SOCKET | SvrAcceptTimeout | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm struct sockaddr * | pSockName | DESCRIZIONE
// @parm int | *iNameLen | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
SOCKET          SvrAcceptTimeout(SOCKET SockFD, struct sockaddr * pSockName, int *iNameLen,
                        int iTimeout)
{

    SOCKET          SockFDAccept;
    int             iConnectError;
    WSAEVENT        hAcceptEvent;

    if ((hAcceptEvent = WSACreateEvent()) == NULL)
        return (INVALID_SOCKET);

    WSAEventSelect(SockFD, hAcceptEvent, FD_ACCEPT);

    SockFDAccept = WSAAccept(SockFD, pSockName, iNameLen, NULL, 0);

    iConnectError = WSAGetLastError();

    if ((SockFDAccept == INVALID_SOCKET) && (iConnectError == WSAEWOULDBLOCK))
    {
        DWORD           dwWaitResult,
                        dwEntryCount = GetTickCount();

        while ((dwWaitResult = WSAWaitForMultipleEvents(1, &hAcceptEvent, FALSE,
                                dwStepTimeout, TRUE)) != WSA_WAIT_EVENT_0)
        {
            if (dwWaitResult == WSA_WAIT_FAILED)
                break;

            if ((iTimeout > 0) &&
                    ((iTimeout * 1000) < (int) (GetTickCount() - dwEntryCount)))
                break;

            if (CallIdleProc() < 0)
                break;
        }

        if (dwWaitResult == WSA_WAIT_EVENT_0)
            SockFDAccept = WSAAccept(SockFD, pSockName, iNameLen, NULL, 0);
    }

    WSAEventSelect(SockFD, hAcceptEvent, 0);
    WSACloseEvent(hAcceptEvent);

    if (SockFDAccept != INVALID_SOCKET)
    {
        if (SvrSetBlocking(SockFDAccept, FALSE) == SOCKET_ERROR)
        {
            SvrCloseSocket(SockFDAccept, 1);
            return ((SOCKET) INVALID_SOCKET);
        }
    }

    return (SockFDAccept);

}




//
// Funzione SVRSEND
//
// @func int | SvrSend | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm BYTE * | pBuffer | DESCRIZIONE
// @parm int | iSize | DESCRIZIONE
// @parm int | iFlags | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
int             SvrSend(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags, int iTimeout)
{

    int             iRtxTotal = 0;

    while (iRtxTotal < iSize)
    {
        int             iRtxSize = min(iSize - iRtxTotal, iRTXBlockSize);
        DWORD           dwEntryCount = GetTickCount();

        int             iBytesSended = SvrSendLL(SockFD, pBuffer + iRtxTotal,
                iRtxSize, iFlags);

        if (iBytesSended > 0)
            iRtxTotal += iBytesSended;

        if (iBytesSended != iRtxSize)
            return (iRtxTotal);

        if ((iTimeout > 0) &&
                ((iTimeout * 1000) < (int) (GetTickCount() - dwEntryCount)))
            return (iRtxTotal);
    }

    return (iRtxTotal);

}




//
// Funzione SVRRECV
//
// @func int | SvrRecv | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm BYTE * | pBuffer | DESCRIZIONE
// @parm int | iSize | DESCRIZIONE
// @parm int | iFlags | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
int             SvrRecv(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags, int iTimeout)
{

    int             iRtxTotal = 0;

    while (iRtxTotal < iSize)
    {
        int             iRtxSize = min(iSize - iRtxTotal, iRTXBlockSize);
        DWORD           dwEntryCount = GetTickCount();

        int             iBytesReceived = SvrRecvLL(SockFD, pBuffer + iRtxTotal,
                iRtxSize, iFlags);

        if (iBytesReceived > 0)
            iRtxTotal += iBytesReceived;

        if (iBytesReceived != iRtxSize)
            return (iRtxTotal);

        if ((iTimeout > 0) &&
                ((iTimeout * 1000) < (int) (GetTickCount() - dwEntryCount)))
            return (iRtxTotal);
    }

    return (iRtxTotal);

}





//
// Funzione SVRWAITASYNCRESPONSE
//
// @func static int | SvrWaitAsyncResponse | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm WSAEVENT | hRtxEvent | DESCRIZIONE
// @parm WSAOVERLAPPED * | pOVLP | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
// MadeBy : Davide Libenzi
//
static int      SvrWaitAsyncResponse(SOCKET SockFD, WSAEVENT hRtxEvent, WSAOVERLAPPED * pOVLP,
                        int iTimeout)
{

    DWORD           dwEntryCount = GetTickCount(),
                    dwWaitResult,
                    dwRtxBytes = 0,
                    dwRtxFlags = 0;

    while ((dwWaitResult = WSAWaitForMultipleEvents(1, &hRtxEvent, FALSE,
                            dwStepTimeout, TRUE)) != WSA_WAIT_EVENT_0)
    {
        if (dwWaitResult == WSA_WAIT_FAILED)
            return (0);

        if (WSAGetOverlappedResult(SockFD, pOVLP, &dwRtxBytes, FALSE, &dwRtxFlags))
            return ((int) dwRtxBytes);

        if ((WSAGetLastError() != WSA_IO_INCOMPLETE) &&
                (WSAGetLastError() != WSA_IO_PENDING))
            return (0);

        if (CallIdleProc() < 0)
            return (0);

        if ((iTimeout > 0) &&
                ((iTimeout * 1000) < (int) (GetTickCount() - dwEntryCount)))
            return (0);
    }

    return ((WSAGetOverlappedResult(SockFD, pOVLP, &dwRtxBytes, TRUE, &dwRtxFlags)) ?
            (int) dwRtxBytes : 0);

}





//
// Funzione SVRSENDLL
//
// @func static int | SvrSendLL | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm BYTE * | pBuffer | DESCRIZIONE
// @parm int | iSize | DESCRIZIONE
// @parm int | iFlags | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
static int      SvrSendLL(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags)
{

    int             iRtxTotal = 0;
    WSAEVENT        hRtxEvent;

    if ((hRtxEvent = WSACreateEvent()) == WSA_INVALID_EVENT)
        return (-1);

    while (iRtxTotal < iSize)
    {
        int             iRtxResult,
                        iRtxError;
        DWORD           dwRtxBytes = 0,
                        dwRtxFlags = (DWORD) iFlags;
        WSAOVERLAPPED   OVLP;
        WSABUF          WSABuff;

        if (CallIdleProc() < 0)
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }

        ZeroMemory(&OVLP, sizeof(OVLP));
        OVLP.hEvent = hRtxEvent;

        ZeroMemory(&WSABuff, sizeof(WSABuff));
        WSABuff.len = iSize - iRtxTotal;
        WSABuff.buf = (char *) (pBuffer + iRtxTotal);

        iRtxResult = WSASend(SockFD, &WSABuff, 1, &dwRtxBytes, dwRtxFlags, &OVLP, NULL);

        iRtxError = WSAGetLastError();

        if (iRtxResult == 0)
        {
            iRtxTotal += (int) dwRtxBytes;
            continue;
        }
        else if (iRtxError == WSA_IO_PENDING)
        {
            int             iAsyncBytes = SvrWaitAsyncResponse(SockFD, hRtxEvent, &OVLP);

            if (iAsyncBytes <= 0)
            {
                WSACloseEvent(hRtxEvent);
                return (iRtxTotal);
            }

            iRtxTotal += iAsyncBytes;
        }
        else
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }
    }

    WSACloseEvent(hRtxEvent);

    return (iRtxTotal);

}







//
// Funzione SVRSENDTO
//
// @func int | SvrSendTo | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm BYTE * | pBuffer | DESCRIZIONE
// @parm int | iSize | DESCRIZIONE
// @parm int | iFlags | DESCRIZIONE
// @parm const struct sockaddr FAR * | lpTo | DESCRIZIONE
// @parm int | iToLen | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
int             SvrSendTo(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags,
                        const struct sockaddr FAR * lpTo, int iToLen)
{

    int             iRtxTotal = 0;
    WSAEVENT        hRtxEvent;

    if ((hRtxEvent = WSACreateEvent()) == WSA_INVALID_EVENT)
        return (-1);

    while (iRtxTotal < iSize)
    {
        int             iRtxResult,
                        iRtxError;
        DWORD           dwRtxBytes = 0,
                        dwRtxFlags = (DWORD) iFlags;
        WSAOVERLAPPED   OVLP;
        WSABUF          WSABuff;

        if (CallIdleProc() < 0)
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }

        ZeroMemory(&OVLP, sizeof(OVLP));
        OVLP.hEvent = hRtxEvent;

        ZeroMemory(&WSABuff, sizeof(WSABuff));
        WSABuff.len = iSize - iRtxTotal;
        WSABuff.buf = (char *) (pBuffer + iRtxTotal);

        iRtxResult = WSASendTo(SockFD, &WSABuff, 1, &dwRtxBytes, dwRtxFlags,
                lpTo, iToLen, &OVLP, NULL);

        iRtxError = WSAGetLastError();

        if (iRtxResult == 0)
        {
            iRtxTotal += (int) dwRtxBytes;
            continue;
        }
        else if (iRtxError == WSA_IO_PENDING)
        {
            int             iAsyncBytes = SvrWaitAsyncResponse(SockFD, hRtxEvent, &OVLP);

            if (iAsyncBytes <= 0)
            {
                WSACloseEvent(hRtxEvent);
                return (iRtxTotal);
            }

            iRtxTotal += iAsyncBytes;
        }
        else
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }
    }

    WSACloseEvent(hRtxEvent);

    return (iRtxTotal);

}






//
// Funzione SVRRECVLL
//
// @func static int | SvrRecvLL | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm BYTE * | pBuffer | DESCRIZIONE
// @parm int | iSize | DESCRIZIONE
// @parm int | iFlags | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
static int      SvrRecvLL(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags)
{

    int             iRtxTotal = 0;
    WSAEVENT        hRtxEvent;

    if ((hRtxEvent = WSACreateEvent()) == WSA_INVALID_EVENT)
        return (-1);

    while (iRtxTotal < iSize)
    {
        int             iRtxResult,
                        iRtxError;
        DWORD           dwRtxBytes = 0,
                        dwRtxFlags = (DWORD) iFlags;
        WSAOVERLAPPED   OVLP;
        WSABUF          WSABuff;

        if (CallIdleProc() < 0)
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }

        ZeroMemory(&OVLP, sizeof(OVLP));
        OVLP.hEvent = hRtxEvent;

        ZeroMemory(&WSABuff, sizeof(WSABuff));
        WSABuff.len = iSize - iRtxTotal;
        WSABuff.buf = (char *) (pBuffer + iRtxTotal);

        iRtxResult = WSARecv(SockFD, &WSABuff, 1, &dwRtxBytes, &dwRtxFlags, &OVLP, NULL);

        iRtxError = WSAGetLastError();

        if (iRtxResult == 0)
        {
            iRtxTotal += (int) dwRtxBytes;
            continue;
        }
        else if (iRtxError == WSA_IO_PENDING)
        {
            int             iAsyncBytes = SvrWaitAsyncResponse(SockFD, hRtxEvent, &OVLP);

            if (iAsyncBytes <= 0)
            {
                WSACloseEvent(hRtxEvent);
                return (iRtxTotal);
            }

            iRtxTotal += iAsyncBytes;
        }
        else
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }
    }

    WSACloseEvent(hRtxEvent);

    return (iRtxTotal);

}






//
// Funzione SVRTRYRECV
//
// @func int | SvrTryRecv | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm BYTE * | pBuffer | DESCRIZIONE
// @parm int | iSize | DESCRIZIONE
// @parm int | iFlags | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
// Creatore : Davide Libenzi
//
int             SvrTryRecv(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags,
                        int iTimeout)
{

    WSAEVENT        hRtxEvent;

    if ((hRtxEvent = WSACreateEvent()) == WSA_INVALID_EVENT)
        return (-1);

    WSAOVERLAPPED   OVLP;
    WSABUF          WSABuff;

    ZeroMemory(&OVLP, sizeof(OVLP));
    OVLP.hEvent = hRtxEvent;

    ZeroMemory(&WSABuff, sizeof(WSABuff));
    WSABuff.len = iSize;
    WSABuff.buf = (char *) pBuffer;

    DWORD           dwRtxBytes = 0,
                    dwRtxFlags = (DWORD) iFlags;

    int             iRtxResult = WSARecv(SockFD, &WSABuff, 1, &dwRtxBytes, &dwRtxFlags, &OVLP, NULL),
                    iRtxError = WSAGetLastError();

    if (iRtxResult == 0)
    {
        WSACloseEvent(hRtxEvent);
        return ((int) dwRtxBytes);
    }

    if (iRtxError == WSA_IO_PENDING)
    {
        int             iAsyncBytes = SvrWaitAsyncResponse(SockFD, hRtxEvent, &OVLP);

        WSACloseEvent(hRtxEvent);

        return (iAsyncBytes);
    }

    WSACloseEvent(hRtxEvent);

    return (-1);

}




//
// Funzione SVRRECVFROM
//
// @func int | SvrRecvFrom | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm BYTE * | pBuffer | DESCRIZIONE
// @parm int | iSize | DESCRIZIONE
// @parm int | iFlags | DESCRIZIONE
// @parm struct sockaddr FAR * | lpFrom | DESCRIZIONE
// @parm LPINT | lpFromlen | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
int             SvrRecvFrom(SOCKET SockFD, BYTE * pBuffer, int iSize, int iFlags,
                        struct sockaddr FAR * lpFrom, LPINT lpFromlen)
{

    int             iRtxTotal = 0;
    WSAEVENT        hRtxEvent;

    if ((hRtxEvent = WSACreateEvent()) == WSA_INVALID_EVENT)
        return (-1);

    while (iRtxTotal < iSize)
    {
        int             iRtxResult,
                        iRtxError;
        DWORD           dwRtxBytes = 0,
                        dwRtxFlags = (DWORD) iFlags;
        WSAOVERLAPPED   OVLP;
        WSABUF          WSABuff;

        if (CallIdleProc() < 0)
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }

        ZeroMemory(&OVLP, sizeof(OVLP));
        OVLP.hEvent = hRtxEvent;

        ZeroMemory(&WSABuff, sizeof(WSABuff));
        WSABuff.len = iSize - iRtxTotal;
        WSABuff.buf = (char *) (pBuffer + iRtxTotal);

        iRtxResult = WSARecvFrom(SockFD, &WSABuff, 1, &dwRtxBytes, &dwRtxFlags,
                lpFrom, lpFromlen, &OVLP, NULL);

        iRtxError = WSAGetLastError();

        if (iRtxResult == 0)
        {
            iRtxTotal += (int) dwRtxBytes;
            continue;
        }
        else if (iRtxError == WSA_IO_PENDING)
        {
            int             iAsyncBytes = SvrWaitAsyncResponse(SockFD, hRtxEvent, &OVLP);

            if (iAsyncBytes <= 0)
            {
                WSACloseEvent(hRtxEvent);
                return (iRtxTotal);
            }

            iRtxTotal += iAsyncBytes;
        }
        else
        {
            WSACloseEvent(hRtxEvent);
            return (iRtxTotal);
        }
    }

    WSACloseEvent(hRtxEvent);

    return (iRtxTotal);

}





//
// Funzione SVRCREATESOCKET
//
// @func SOCKET | SvrCreateSocket | DESCRIZIONE
//
// @parm int | iAddressFamily | DESCRIZIONE
// @parm int | iType | DESCRIZIONE
// @parm int | iProtocol | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
SOCKET          SvrCreateSocket(int iAddressFamily, int iType, int iProtocol)
{

    SOCKET          SockFD;

    if ((SockFD = WSASocket(iAddressFamily, iType, iProtocol, NULL, 0,
                            WSA_FLAG_OVERLAPPED)) == INVALID_SOCKET)
        return (INVALID_SOCKET);

    if (SvrSetBlocking(SockFD, FALSE) == SOCKET_ERROR)
    {
        closesocket(SockFD);
        return (INVALID_SOCKET);
    }

    return (SockFD);

}






//
// Funzione SVRCLOSESOCKET
//
// @func int | SvrCloseSocket | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm int | iHardClose | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
// Creatore : Davide Libenzi
//
int             SvrCloseSocket(SOCKET SockFD, int iHardClose)
{

    if (!iHardClose)
    {
        shutdown(SockFD, SD_SEND);

        int             iRecvResult;

        do
        {
            char            szBuffer[256] = "";

            iRecvResult = SvrTryRecv(SockFD, (BYTE *) szBuffer, sizeof(szBuffer), 0,
                    SHUTDOWN_RECV_TIMEOUT);

        } while (iRecvResult > 0);

        shutdown(SockFD, SD_BOTH);
    }

    closesocket(SockFD);

    return (0);

}




//
// Funzione SVRCONNECTSERVER
//
// @func SOCKET | SvrConnectServer | DESCRIZIONE
//
// @parm char | *pszSvrNode | DESCRIZIONE
// @parm int | iSvrPort | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
SOCKET          SvrConnectServer(char *pszSvrNode, int iSvrPort, int iTimeout)
{

    SOCKET          SockFD;
    struct hostent *pHostEnt = NULL;
    struct sockaddr_in SvrAddr;

    if ((SockFD = SvrCreateSocket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
        return ((SOCKET) INVALID_SOCKET);

    ZeroMemory(&SvrAddr, sizeof(SvrAddr));
    SvrAddr.sin_family = AF_INET;

    if (((SvrAddr.sin_addr.s_addr = inet_addr(pszSvrNode)) == INADDR_NONE) &&
            ((pHostEnt = gethostbyname(pszSvrNode)) != NULL))
        SvrAddr.sin_addr.s_addr = *((unsigned long int *) pHostEnt->h_addr_list[0]);

    SvrAddr.sin_port = htons((u_short) iSvrPort);

    if (SvrConnectTimeout(SockFD, (struct sockaddr *) & SvrAddr, sizeof(SvrAddr),
                    iTimeout) < 0)
    {
        SvrCloseSocket(SockFD, 1);
        return ((SOCKET) INVALID_SOCKET);
    }

    return (SockFD);

}






//
// Funzione SVRSENDDWORD
//
// @func BOOL | SvrSendDWORD | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm DWORD | dwData | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
BOOL            SvrSendDWORD(SOCKET SockFD, DWORD dwData, int iTimeout)
{

    DWORD           dwNetData;

    dwNetData = (DWORD) htonl(dwData);

    return ((SvrSend(SockFD, (BYTE *) & dwNetData, sizeof(dwNetData), 0,
                            iTimeout) == sizeof(dwNetData)) ? TRUE : FALSE);

}






//
// Funzione SVRRECVDWORD
//
// @func BOOL | SvrRecvDWORD | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm DWORD * | pdwData | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
BOOL            SvrRecvDWORD(SOCKET SockFD, DWORD * pdwData, int iTimeout)
{

    DWORD           dwNetData;

    if (SvrRecv(SockFD, (BYTE *) & dwNetData, sizeof(dwNetData), 0,
                    iTimeout) != sizeof(dwNetData))
        return (FALSE);

    *pdwData = (DWORD) ntohl(dwNetData);

    return (TRUE);

}






//
// Funzione SVRSENDWORD
//
// @func BOOL | SvrSendWORD | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm WORD | wData | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
BOOL            SvrSendWORD(SOCKET SockFD, WORD wData, int iTimeout)
{

    WORD            dwNetData;

    dwNetData = (WORD) htons(wData);

    return ((SvrSend(SockFD, (BYTE *) & dwNetData, sizeof(dwNetData), 0,
                            iTimeout) == sizeof(dwNetData)) ? TRUE : FALSE);

}






//
// Funzione SVRRECVWORD
//
// @func BOOL | SvrRecvWORD | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm WORD * | pwData | DESCRIZIONE
// @parm int | iTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
BOOL            SvrRecvWORD(SOCKET SockFD, WORD * pwData, int iTimeout)
{

    WORD            dwNetData;

    if (SvrRecv(SockFD, (BYTE *) & dwNetData, sizeof(dwNetData), 0,
                    iTimeout) != sizeof(dwNetData))
        return (FALSE);

    *pwData = (WORD) ntohs(dwNetData);

    return (TRUE);

}






//
// Funzione SVRGETCHILDSOCKET
//
// @func SOCKET | SvrGetChildSocket | DESCRIZIONE
//
// @parm SOCKET | SockFD | DESCRIZIONE
// @parm int | iStartPortNo | DESCRIZIONE
// @parm int | iEndPortNo | DESCRIZIONE
// @parm int | iAcceptTimeout | DESCRIZIONE
// @parm int | iSendTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
SOCKET          SvrGetChildSocket(SOCKET SockFD, int iStartPortNo, int iEndPortNo,
                        int iAcceptTimeout, int iSendTimeout)
{

    int             iCliLen,
                    iChildPort;
    SOCKET          ChildSockFD,
                    AcceptSockFD;
    struct sockaddr_in CliAddr;

    if ((ChildSockFD = SvrCreateSocket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
        return (INVALID_SOCKET);

    for (iChildPort = iStartPortNo; iChildPort < iEndPortNo; iChildPort++)
    {
        ZeroMemory(&CliAddr, sizeof(CliAddr));
        CliAddr.sin_family = AF_INET;
        CliAddr.sin_addr.s_addr = htonl(INADDR_ANY);
        CliAddr.sin_port = htons((u_short) iChildPort);

        if (bind(ChildSockFD, (struct sockaddr *) & CliAddr, sizeof(CliAddr)) == 0)
            break;
    }

    if (iChildPort == iEndPortNo)
    {
        SvrCloseSocket(ChildSockFD, 1);
        return (INVALID_SOCKET);
    }

    if (!SvrSendDWORD(SockFD, (DWORD) iChildPort, iSendTimeout))
    {
        SvrCloseSocket(ChildSockFD, 1);
        return (INVALID_SOCKET);
    }

    listen(ChildSockFD, 1);

    ZeroMemory(&CliAddr, sizeof(CliAddr));
    iCliLen = sizeof(CliAddr);

    if ((AcceptSockFD = SvrAcceptTimeout(ChildSockFD, (struct sockaddr *) & CliAddr,
                            &iCliLen, iAcceptTimeout)) == INVALID_SOCKET)
    {
        SvrCloseSocket(ChildSockFD, 1);
        return (INVALID_SOCKET);
    }

    SvrCloseSocket(ChildSockFD, 1);

    return (AcceptSockFD);

}






//
// Funzione SVRSWITCHCONNECTSERVER
//
// @func SOCKET | SvrSwitchConnectServer | DESCRIZIONE
//
// @parm char | *pszSvrNode | DESCRIZIONE
// @parm int | iSvrPort | DESCRIZIONE
// @parm int | iConnectTimeout | DESCRIZIONE
// @parm int | iRecvTimeout | DESCRIZIONE
//
// @rdesc DESCRIZIONE_VALORE_RITORNO
//
SOCKET          SvrSwitchConnectServer(char *pszSvrNode, int iSvrPort, int iConnectTimeout,
                        int iRecvTimeout)
{

    SOCKET          SockFD;
    DWORD           dwChildPort = 0;
    unsigned long   ulNetAddr = INADDR_NONE;
    struct hostent *pHostEnt = NULL;
    struct sockaddr_in SvrAddr;

    if (((ulNetAddr = inet_addr(pszSvrNode)) == INADDR_NONE) &&
            ((pHostEnt = gethostbyname(pszSvrNode)) != NULL))
        ulNetAddr = *((unsigned long int *) pHostEnt->h_addr_list[0]);

    if (ulNetAddr == INADDR_NONE)
        return ((SOCKET) INVALID_SOCKET);

    if ((SockFD = SvrCreateSocket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
        return ((SOCKET) INVALID_SOCKET);

    ZeroMemory(&SvrAddr, sizeof(SvrAddr));
    SvrAddr.sin_family = AF_INET;
    SvrAddr.sin_addr.s_addr = ulNetAddr;
    SvrAddr.sin_port = htons((u_short) iSvrPort);

    if (SvrConnectTimeout(SockFD, (struct sockaddr *) & SvrAddr, sizeof(SvrAddr),
                    iConnectTimeout) < 0)
    {
        SvrCloseSocket(SockFD, 1);
        return ((SOCKET) INVALID_SOCKET);
    }

    if (!SvrRecvDWORD(SockFD, &dwChildPort, iRecvTimeout))
    {
        SvrCloseSocket(SockFD, 1);
        return ((SOCKET) INVALID_SOCKET);
    }

    SvrCloseSocket(SockFD, 1);

    if ((SockFD = SvrCreateSocket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
        return ((SOCKET) INVALID_SOCKET);

    ZeroMemory(&SvrAddr, sizeof(SvrAddr));
    SvrAddr.sin_family = AF_INET;
    SvrAddr.sin_addr.s_addr = ulNetAddr;
    SvrAddr.sin_port = htons((u_short) dwChildPort);

    if (SvrConnectTimeout(SockFD, (struct sockaddr *) & SvrAddr, sizeof(SvrAddr),
                    iConnectTimeout) < 0)
    {
        SvrCloseSocket(SockFD, 1);
        return ((SOCKET) INVALID_SOCKET);
    }

    return (SockFD);

}
