/*
 *  SndMail by Davide Libenzi ( SMTP Internet mail delivery agent )
 *  Copyright (C) 2000, 2001  Davide Libenzi
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Davide Libenzi <davidel@xmailserver.org>
 *
 */



#include <winsock2.h>
#include <mswsock.h>
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include <ctype.h>
#include <limits.h>
#include <time.h>
#include "sockio.h"
#include "BuffSock.h"
#include "Base64Enc.h"
#include "MD5.h"
#include "SndMail.h"






#define MAX_RCPT                        128
#define MAX_CC                          128
#define MAX_TAGS                        64
#define MAX_ATTACH                      64


#define ZeroData(s)                     memset(&(s), 0, sizeof(s))
#define ArraySize(a)                    (sizeof(a) / sizeof((a)[0]))
#define Sign(v)                         (((v) < 0) ? -1: +1)
#define Min(a, b)                       (((a) < (b)) ? (a): (b))
#define Max(a, b)                       (((a) > (b)) ? (a): (b))
#define Abs(v)                          (((v) > 0) ? (v): -(v))




struct ErrorInfo
{
    int             iCode;
    char const     *pszError;
    char const     *pszInfo;
};







static int      USmtpGetTime(struct tm & tmLocal, int &iDiffHours, int &iDiffMins);
static int      USmtpGetTimeStr(char *pszTimeStr, int iStringSize);
static void     ErrFreeErrorsInfo(void);
static ErrorInfo *ErrGetErrorInfo(int iErrorCode);
static char const *ErrGetErrorString(int iErrorCode);
static int      ErrSetErrorCode(int iErrorCode, char const * pszInfo = NULL);
static int      ErrGetErrorCode(void);
static int      USmtpCmdLineToken(char const * &pszCmdLine, char *pszToken);
static char   **USmtpGetArgs(char const * pszCmdLine, int &iArgsCount);
static void     USmtpFreeArgs(char **ppszArgs);
static char    *USmtpGetString(FILE * pFile, char *pszBuffer, int iBufferSize);
static int      USmtpResponseClass(int iResponseCode, int iResponseClass);
static int      USmtpGetResultCode(char const * pszResult);
static int      USmtpIsPartialResponse(char const * pszResponse);
static int      USmtpGetResponse(BSOCK_HANDLE hBSock, char *pszResponse, int iMaxResponse,
                        int iTimeout = STD_SMTP_TIMEOUT);
static int      USmtpSendCommand(BSOCK_HANDLE hBSock, char const * pszCommand,
                        char *pszResponse, int iMaxResponse, int iTimeout = STD_SMTP_TIMEOUT);
static int      USmtpSendFile(char const * pszFileName, BSOCK_HANDLE hBSock, int iParseFile);
static int      USmtpDoPlainAuth(BSOCK_HANDLE hBSock, char const * pszUsername,
                        char const * pszPassword);
static int      USmtpDoLoginAuth(BSOCK_HANDLE hBSock, char const * pszUsername,
                        char const * pszPassword);
static int      USmtpCramMD5(char const * pszSecret, char const * pszChallenge,
                        char *pszDigest);
static int      USmtpDoCramMD5Auth(BSOCK_HANDLE hBSock, char const * pszUsername,
                        char const * pszPassword);
static int      USmtpAuthenticate(BSOCK_HANDLE hBSock, MailSendData const * pMSD);
static BSOCK_HANDLE USmtpCreateChannel(char const * pszServer, int iPortNo, char const * pszDomain,
                        MailSendData const * pMSD);
static int      USmtpCloseChannel(BSOCK_HANDLE hBSock, int iHardClose = 0);
static int      USmtpChannelReset(BSOCK_HANDLE hBSock);
static int      USmtpFilePrepare(char const * pszMailFile, MailSendData const * pMSD);
static int      USmtpGetTempFile(char *pszTempFile);
static int      USmtpSendMail(BSOCK_HANDLE hBSock, MailSendData const * pMSD);
static int      USmtpCheckMSD(MailSendData const * pMSD);





static ErrorInfo USmtpErrors[] =
{
    {ERR_BAD_SMTP_RESPONSE, "Wrong SMTP server response", NULL},
    {ERR_BAD_SERVER_RESPONSE, "SMTP server error response", NULL},
    {ERR_BAD_SERVER_ADDR, "Bad SMTP server address", NULL},
    {ERR_BAD_SMTP_RESPONSE, "Bad SMTP server response", NULL},
    {ERR_FILE_CREATE, "File create error", NULL},
    {ERR_FILE_OPEN, "File open error", NULL},
    {ERR_INVALID_MSD_VERSION, "Wrong MailSendData version", NULL},
    {ERR_MSD_NO_BODYFILE, "Body file name not specified", NULL},
    {ERR_MSD_NO_SUBJECT, "Subject not specified", NULL},
    {ERR_MSD_NO_FROM, "From not specified", NULL},
    {ERR_MSD_NO_RCPT, "Rcpt not specified", NULL},
    {ERR_NO_SOCK_SUPPORT, "Socket support not found", NULL},
    {ERR_UNKNOWN_SWITCH, "Unknown switch", NULL},
    {ERR_INVALID_ERROR_CODE, "Invalid error code", NULL},
    {ERR_NO_MORE_TOKENS, "No more tokens", NULL},
    {ERR_MEMORY_ALLOCATION, "Memory allocation error", NULL},
    {ERR_NO_AUTH_INFO, "Authentication infos not supplied", NULL},

};


static int      iErrorNo = 0,
                iLibOpen = 0;







static int      USmtpGetTime(struct tm & tmLocal, int &iDiffHours, int &iDiffMins)
{

    time_t          tCurr;

    time(&tCurr);

    tmLocal = *localtime(&tCurr);


    struct tm       tmTimeLOC = tmLocal,
                    tmTimeGM = *gmtime(&tCurr);

    tmTimeLOC.tm_isdst = 0;
    tmTimeGM.tm_isdst = 0;

    time_t          tLocal = mktime(&tmTimeLOC),
                    tGM = mktime(&tmTimeGM);

    int             iSecsDiff = (int) difftime(tLocal, tGM),
                    iSignDiff = Sign(iSecsDiff),
                    iMinutes = Abs(iSecsDiff) / 60;

    iDiffMins = iMinutes % 60;
    iDiffHours = iSignDiff * (iMinutes / 60);


    return (0);

}




static int      USmtpGetTimeStr(char *pszTimeStr, int iStringSize)
{

    int             iDiffHours = 0,
                    iDiffMins = 0;
    struct tm       tmTime;

    USmtpGetTime(tmTime, iDiffHours, iDiffMins);


    char            szDiffTime[128] = "";

    if (iDiffHours > 0)
        sprintf(szDiffTime, " +%02d%02d", iDiffHours, iDiffMins);
    else
        sprintf(szDiffTime, " -%02d%02d", -iDiffHours, iDiffMins);


    strftime(pszTimeStr, iStringSize - strlen(szDiffTime) - 1, "%a, %d %b %Y %H:%M:%S", &tmTime);

    strcat(pszTimeStr, szDiffTime);


    return (0);

}




static void     ErrFreeErrorsInfo(void)
{
    for (int ii = 0; ii < ArraySize(USmtpErrors); ii++)
        if (USmtpErrors[ii].pszInfo != NULL)
            free((char *) USmtpErrors[ii].pszInfo), USmtpErrors[ii].pszInfo = NULL;

}



static ErrorInfo *ErrGetErrorInfo(int iErrorCode)
{

    for (int ii = 0; ii < ArraySize(USmtpErrors); ii++)
        if (USmtpErrors[ii].iCode == iErrorCode)
            return (&USmtpErrors[ii]);

    return (NULL);

}



static char const *ErrGetErrorString(int iErrorCode)
{

    for (int ii = 0; ii < ArraySize(USmtpErrors); ii++)
        if (USmtpErrors[ii].iCode == iErrorCode)
            return (USmtpErrors[ii].pszError);

    return ("????");

}



static int      ErrSetErrorCode(int iErrorCode, char const * pszInfo)
{

    if (pszInfo != NULL)
    {
        ErrorInfo      *pEI = ErrGetErrorInfo(iErrorCode);

        if (pEI != NULL)
        {
            if (pEI->pszInfo != NULL)
                free((char *) pEI->pszInfo);

            pEI->pszInfo = strdup(pszInfo);
        }
    }

    return (iErrorNo = iErrorCode);

}



static int      ErrGetErrorCode(void)
{

    return (iErrorNo);

}



static int      USmtpCmdLineToken(char const * &pszCmdLine, char *pszToken)
{

    char const     *pszCurr = pszCmdLine;

    for (; (*pszCurr == ' ') || (*pszCurr == '\t'); pszCurr++);

    if (*pszCurr == '\0')
        return (ERR_NO_MORE_TOKENS);

    if (*pszCurr == '"')
    {
        ++pszCurr;

        for (; *pszCurr != '\0';)
        {
            if (*pszCurr == '"')
            {
                ++pszCurr;

                if (*pszCurr != '"')
                    break;

                *pszToken++ = *pszCurr++;
            }
            else
                *pszToken++ = *pszCurr++;
        }

        *pszToken = '\0';
    }
    else
    {
        for (; (*pszCurr != ' ') && (*pszCurr != '\t') && (*pszCurr != '\0');)
            *pszToken++ = *pszCurr++;

        *pszToken = '\0';
    }

    pszCmdLine = pszCurr;

    return (0);

}



static char   **USmtpGetArgs(char const * pszCmdLine, int &iArgsCount)
{

    char const     *pszCLine = pszCmdLine;
    char            szToken[1024] = "";

    for (iArgsCount = 0; USmtpCmdLineToken(pszCLine, szToken) == 0; iArgsCount++);


    char          **ppszArgs = (char **) malloc((iArgsCount + 1) * sizeof(char *));

    if (ppszArgs == NULL)
    {
        ErrSetErrorCode(ERR_MEMORY_ALLOCATION);
        return (NULL);
    }

    int             ii = 0;

    for (pszCLine = pszCmdLine; (ii < iArgsCount) && (USmtpCmdLineToken(pszCLine, szToken) == 0); ii++)
        ppszArgs[ii] = strdup(szToken);

    ppszArgs[ii] = NULL;

    return (ppszArgs);

}



static void     USmtpFreeArgs(char **ppszArgs)
{

    for (int ii = 0; ppszArgs[ii] != NULL; ii++)
        free(ppszArgs[ii]);

    free(ppszArgs);

}



static char    *USmtpGetString(FILE * pFile, char *pszBuffer, int iBufferSize)
{

    if (fgets(pszBuffer, iBufferSize, pFile) == NULL)
        return (NULL);

    pszBuffer[strlen(pszBuffer) - 1] = '\0';

    return (pszBuffer);

}



static int      USmtpResponseClass(int iResponseCode, int iResponseClass)
{

    return (((iResponseCode >= iResponseClass) &&
                    (iResponseCode < (iResponseClass + 100))) ? 1 : 0);

}



static int      USmtpGetResultCode(char const * pszResult)
{

    int             ii;
    char            szResCode[64] = "";

    for (ii = 0; isdigit(pszResult[ii]); ii++)
        szResCode[ii] = pszResult[ii];

    szResCode[ii] = '\0';

    if (ii == 0)
    {
        ErrSetErrorCode(ERR_BAD_SMTP_RESPONSE, pszResult);
        return (ERR_BAD_SMTP_RESPONSE);
    }

    return (atoi(szResCode));

}



static int      USmtpIsPartialResponse(char const * pszResponse)
{

    return (((strlen(pszResponse) >= 4) && (pszResponse[3] == '-')) ? 1 : 0);

}



static int      USmtpGetResponse(BSOCK_HANDLE hBSock, char *pszResponse, int iMaxResponse,
                        int iTimeout)
{

    int             iResultCode = -1,
                    iResponseLenght = 0;
    char            szPartial[1024] = "";

    do
    {
        if (BSckGetString(hBSock, szPartial, sizeof(szPartial) - 1, iTimeout) == NULL)
            return (ErrGetErrorCode());

        if (iResponseLenght < iMaxResponse)
        {
            if (iResponseLenght > 0)
                strcat(pszResponse, "\r\n"), iResponseLenght += 2;

            int             iCopyLenght = min(iMaxResponse - 1 - iResponseLenght,
                    (int) strlen(szPartial));

            strncpy(pszResponse + iResponseLenght, szPartial, iCopyLenght);

            iResponseLenght += iCopyLenght;

            pszResponse[iResponseLenght] = '\0';
        }

        if ((iResultCode = USmtpGetResultCode(szPartial)) < 0)
            return (ErrGetErrorCode());

    } while (USmtpIsPartialResponse(szPartial));

    return (iResultCode);

}



static int      USmtpSendCommand(BSOCK_HANDLE hBSock, char const * pszCommand,
                        char *pszResponse, int iMaxResponse, int iTimeout)
{

    if (BSckSendString(hBSock, (char *) pszCommand, iTimeout) <= 0)
        return (ErrGetErrorCode());

    return (USmtpGetResponse(hBSock, pszResponse, iMaxResponse, iTimeout));

}



static int      USmtpSendFile(char const * pszFileName, BSOCK_HANDLE hBSock,
                        int iParseFile)
{

    FILE           *pFile = fopen(pszFileName, "rt");

    if (pFile == NULL)
        return (ErrGetErrorCode());

    char            szBuffer[1024] = "";

    while (USmtpGetString(pFile, szBuffer, sizeof(szBuffer) - 1) != NULL)
    {
        if (iParseFile && (szBuffer[0] == '.'))
            for (int ii = strlen(szBuffer); ii >= 0; ii--)
                szBuffer[ii + 1] = szBuffer[ii];

        if (BSckSendString(hBSock, szBuffer, STD_SMTP_TIMEOUT) <= 0)
        {
            fclose(pFile);
            return (ErrGetErrorCode());
        }
    }

    fclose(pFile);

    return (0);

}



static int      USmtpDoPlainAuth(BSOCK_HANDLE hBSock, char const * pszUsername,
                        char const * pszPassword)
{

///////////////////////////////////////////////////////////////////////////////
//  Build plain text authentication token ( "\0" Username "\0" Password "\0" )
///////////////////////////////////////////////////////////////////////////////
    int             iAuthLength = 1;
    char            szAuthBuffer[2048] = "";

    strcpy(szAuthBuffer + iAuthLength, pszUsername);
    iAuthLength += strlen(pszUsername) + 1;

    strcpy(szAuthBuffer + iAuthLength, pszPassword);
    iAuthLength += strlen(pszPassword);

    unsigned int    uEnc64Length = 0;
    char            szEnc64Token[1024] = "";

    encode64(szAuthBuffer, iAuthLength, szEnc64Token, sizeof(szEnc64Token), &uEnc64Length);

///////////////////////////////////////////////////////////////////////////////
//  Send AUTH command
///////////////////////////////////////////////////////////////////////////////
    int             iSvrReponse;

    sprintf(szAuthBuffer, "AUTH PLAIN %s", szEnc64Token);

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szAuthBuffer, szAuthBuffer,
                            sizeof(szAuthBuffer)), 200))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szAuthBuffer);

        return (ErrGetErrorCode());
    }


    return (0);

}



static int      USmtpDoLoginAuth(BSOCK_HANDLE hBSock, char const * pszUsername,
                        char const * pszPassword)
{

///////////////////////////////////////////////////////////////////////////////
//  Send AUTH command
///////////////////////////////////////////////////////////////////////////////
    int             iSvrReponse;
    char            szAuthBuffer[1024] = "";

    sprintf(szAuthBuffer, "AUTH LOGIN");

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szAuthBuffer, szAuthBuffer,
                            sizeof(szAuthBuffer)), 300))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szAuthBuffer);

        return (ErrGetErrorCode());
    }

///////////////////////////////////////////////////////////////////////////////
//  Send username
///////////////////////////////////////////////////////////////////////////////
    unsigned int    uEnc64Length = 0;

    encode64(pszUsername, strlen(pszUsername), szAuthBuffer,
            sizeof(szAuthBuffer), &uEnc64Length);

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szAuthBuffer, szAuthBuffer,
                            sizeof(szAuthBuffer)), 300))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szAuthBuffer);

        return (ErrGetErrorCode());
    }

///////////////////////////////////////////////////////////////////////////////
//  Send password
///////////////////////////////////////////////////////////////////////////////
    encode64(pszPassword, strlen(pszPassword), szAuthBuffer,
            sizeof(szAuthBuffer), &uEnc64Length);

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szAuthBuffer, szAuthBuffer,
                            sizeof(szAuthBuffer)), 200))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szAuthBuffer);

        return (ErrGetErrorCode());
    }


    return (0);

}



static int      USmtpCramMD5(char const * pszSecret, char const * pszChallenge,
                        char *pszDigest)
{

    int             iLenght = (int) strlen(pszSecret);
    struct md5_ctx  ctx;
    unsigned char   isecret[64];
    unsigned char   osecret[64];
    unsigned char   md5secret[16];

    if (iLenght > 64)
    {
        md5_init_ctx(&ctx);
        md5_process_bytes(pszSecret, iLenght, &ctx);
        md5_finish_ctx(&ctx, md5secret);

        pszSecret = (char const *) md5secret;
        iLenght = 16;
    }

    ZeroData(isecret);
    memcpy(isecret, pszSecret, iLenght);

    ZeroData(osecret);
    memcpy(osecret, pszSecret, iLenght);

    for (int ii = 0; ii < 64; ii++)
    {
        isecret[ii] ^= 0x36;
        osecret[ii] ^= 0x5c;
    }


    md5_init_ctx(&ctx);
    md5_process_bytes(isecret, 64, &ctx);
    md5_process_bytes((unsigned char *) pszChallenge, (int) strlen(pszChallenge), &ctx);
    md5_finish_ctx(&ctx, md5secret);


    md5_init_ctx(&ctx);
    md5_process_bytes(osecret, 64, &ctx);
    md5_process_bytes(md5secret, 16, &ctx);
    md5_finish_ctx(&ctx, md5secret);

    md5_hex(md5secret, pszDigest);

    return (0);

}




static int      USmtpDoCramMD5Auth(BSOCK_HANDLE hBSock, char const * pszUsername,
                        char const * pszPassword)
{

///////////////////////////////////////////////////////////////////////////////
//  Send AUTH command
///////////////////////////////////////////////////////////////////////////////
    int             iSvrReponse;
    char            szAuthBuffer[1024] = "";

    sprintf(szAuthBuffer, "AUTH CRAM-MD5");

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szAuthBuffer, szAuthBuffer,
                            sizeof(szAuthBuffer)), 300) ||
            (strlen(szAuthBuffer) < 4))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szAuthBuffer);

        return (ErrGetErrorCode());
    }

///////////////////////////////////////////////////////////////////////////////
//  Retrieve server challenge
///////////////////////////////////////////////////////////////////////////////
    unsigned int    uDec64Length = 0;
    char           *pszAuth = szAuthBuffer + 4;
    char            szChallenge[1024] = "";

    if (decode64(pszAuth, strlen(pszAuth), szChallenge, &uDec64Length) != 0)
    {
        ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szAuthBuffer);
        return (ERR_BAD_SERVER_RESPONSE);
    }

///////////////////////////////////////////////////////////////////////////////
//  Compute MD5 response ( secret , challenge , digest )
///////////////////////////////////////////////////////////////////////////////
    if (USmtpCramMD5(pszPassword, szChallenge, szChallenge) < 0)
        return (ErrGetErrorCode());

///////////////////////////////////////////////////////////////////////////////
//  Send response
///////////////////////////////////////////////////////////////////////////////
    unsigned int    uEnc64Length = 0;
    char            szResponse[1024] = "";

    sprintf(szResponse, "%s %s", pszUsername, szChallenge);

    encode64(szResponse, strlen(szResponse), szAuthBuffer,
            sizeof(szAuthBuffer), &uEnc64Length);

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szAuthBuffer, szAuthBuffer,
                            sizeof(szAuthBuffer)), 200))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szAuthBuffer);

        return (ErrGetErrorCode());
    }


    return (0);

}



static int      USmtpAuthenticate(BSOCK_HANDLE hBSock, MailSendData const * pMSD)
{

    if (pMSD->pszAuthType != NULL)
    {
        if (stricmp(pMSD->pszAuthType, "login") == 0)
        {
            if ((pMSD->pszUsername == NULL) || (pMSD->pszPassword == NULL))
            {
                ErrSetErrorCode(ERR_NO_AUTH_INFO);
                return (ERR_NO_AUTH_INFO);
            }

            return (USmtpDoLoginAuth(hBSock, pMSD->pszUsername, pMSD->pszPassword));
        }
        else if (stricmp(pMSD->pszAuthType, "plain") == 0)
        {
            if ((pMSD->pszUsername == NULL) || (pMSD->pszPassword == NULL))
            {
                ErrSetErrorCode(ERR_NO_AUTH_INFO);
                return (ERR_NO_AUTH_INFO);
            }

            return (USmtpDoPlainAuth(hBSock, pMSD->pszUsername, pMSD->pszPassword));
        }
        else if (stricmp(pMSD->pszAuthType, "cram-md5") == 0)
        {
            if (pMSD->pszPassword == NULL)
            {
                ErrSetErrorCode(ERR_NO_AUTH_INFO);
                return (ERR_NO_AUTH_INFO);
            }

            return (USmtpDoCramMD5Auth(hBSock, pMSD->pszUsername, pMSD->pszPassword));
        }

    }

    return (0);

}



static BSOCK_HANDLE USmtpCreateChannel(char const * pszServer, int iPortNo, char const * pszDomain,
                        MailSendData const * pMSD)
{

    unsigned long   ulNetAddr = INADDR_NONE;
    struct hostent *pHostEnt = NULL;

    if (((ulNetAddr = inet_addr(pszServer)) == INADDR_NONE) &&
            ((pHostEnt = gethostbyname(pszServer)) != NULL))
        ulNetAddr = *((unsigned int *) pHostEnt->h_addr_list[0]);

    if (ulNetAddr == INADDR_NONE)
    {
        ErrSetErrorCode(ERR_BAD_SERVER_ADDR, pszServer);
        return (INVALID_BSOCK_HANDLE);
    }

    SOCKET          SockFD = SvrCreateSocket(AF_INET, SOCK_STREAM, 0);

    if (SockFD == INVALID_SOCKET)
        return (INVALID_BSOCK_HANDLE);


    struct sockaddr_in SvrAddr;

    ZeroData(SvrAddr);
    SvrAddr.sin_family = AF_INET;
    SvrAddr.sin_addr.s_addr = ulNetAddr;
    SvrAddr.sin_port = htons(iPortNo);

    if (SvrConnectTimeout(SockFD, (struct sockaddr *) & SvrAddr, sizeof(SvrAddr),
                    STD_SMTP_TIMEOUT) < 0)
    {
        SvrCloseSocket(SockFD);
        return (INVALID_BSOCK_HANDLE);
    }

    BSOCK_HANDLE    hBSock = BSckAttach(SockFD);

    if (hBSock == INVALID_BSOCK_HANDLE)
    {
        SvrCloseSocket(SockFD);
        return (INVALID_BSOCK_HANDLE);
    }

///////////////////////////////////////////////////////////////////////////////
//  Read welcome message
///////////////////////////////////////////////////////////////////////////////
    int             iSvrReponse = -1;
    char            szRTXBuffer[2048] = "";

    if (!USmtpResponseClass(iSvrReponse = USmtpGetResponse(hBSock, szRTXBuffer,
                            sizeof(szRTXBuffer)), 200))
    {
        BSckDetach(hBSock, 1);
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
        return (INVALID_BSOCK_HANDLE);
    }

///////////////////////////////////////////////////////////////////////////////
//  Send HELO and read result
///////////////////////////////////////////////////////////////////////////////
    if (pMSD->pszAuthType == NULL)
        sprintf(szRTXBuffer, "HELO %s", pszDomain);
    else
        sprintf(szRTXBuffer, "EHLO %s", pszDomain);

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szRTXBuffer, szRTXBuffer,
                            sizeof(szRTXBuffer)), 200))
    {
        USmtpCloseChannel(hBSock);
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
        return (INVALID_BSOCK_HANDLE);
    }

///////////////////////////////////////////////////////////////////////////////
//  Authenticate if needed
///////////////////////////////////////////////////////////////////////////////
    if (USmtpAuthenticate(hBSock, pMSD) < 0)
    {
        USmtpCloseChannel(hBSock);
        return (INVALID_BSOCK_HANDLE);
    }


    return (hBSock);

}



static int      USmtpCloseChannel(BSOCK_HANDLE hBSock, int iHardClose)
{

    if (!iHardClose)
    {
///////////////////////////////////////////////////////////////////////////////
//  Send QUIT and read result
///////////////////////////////////////////////////////////////////////////////
        int             iSvrReponse = -1;
        char            szRTXBuffer[2048] = "";

        if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, "QUIT", szRTXBuffer,
                                sizeof(szRTXBuffer)), 200))
        {
            BSckDetach(hBSock, 1);
            if (iSvrReponse > 0)
                ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
            return (ErrGetErrorCode());
        }
    }

    BSckDetach(hBSock, 1);

    return (0);

}



static int      USmtpChannelReset(BSOCK_HANDLE hBSock)
{

///////////////////////////////////////////////////////////////////////////////
//  Send RSET and read result
///////////////////////////////////////////////////////////////////////////////
    int             iSvrReponse = -1;
    char            szRTXBuffer[2048] = "";

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, "RSET", szRTXBuffer,
                            sizeof(szRTXBuffer)), 200))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
        return (ErrGetErrorCode());
    }

    return (0);

}





static int      USmtpFilePrepare(char const * pszMailFile, MailSendData const * pMSD)
{

    FILE           *pMailFile = fopen(pszMailFile, "wt");

    if (pMailFile == NULL)
    {
        ErrSetErrorCode(ERR_FILE_CREATE, pszMailFile);
        return (ERR_FILE_CREATE);
    }

    if (pMSD->pszFromExt != NULL)
        fprintf(pMailFile, "From: %s\n", pMSD->pszFromExt);
    else
        fprintf(pMailFile, "From: %s\n", pMSD->pszFrom);

    fprintf(pMailFile, "From-Addr: %s\n", pMSD->pszFrom);

    fprintf(pMailFile, "Subject: %s\n", pMSD->pszSubject);

    if (pMSD->pszReply != NULL)
        fprintf(pMailFile, "Reply-To: %s\n", pMSD->pszReply);

    if (pMSD->pszErrors != NULL)
        fprintf(pMailFile, "Errors-To: %s\n", pMSD->pszErrors);

    if (pMSD->pszSender != NULL)
        fprintf(pMailFile, "Sender: %s\n", pMSD->pszSender);


    int             ii;

    fprintf(pMailFile, "To: %s", pMSD->ppszRcpt[0]);

    for (ii = 1; pMSD->ppszRcpt[ii] != NULL; ii++)
        fprintf(pMailFile, ",%s", pMSD->ppszRcpt[ii]);

    fprintf(pMailFile, "\n");


    if (pMSD->ppszTags != NULL)
        for (ii = 0; pMSD->ppszTags[ii] != NULL; ii++)
            fprintf(pMailFile, "%s\n", pMSD->ppszTags[ii]);


    if ((pMSD->ppszCC != NULL) && (pMSD->ppszCC[0] != NULL))
    {
        fprintf(pMailFile, "Cc: %s", pMSD->ppszCC[0]);

        for (ii = 1; pMSD->ppszCC[ii] != NULL; ii++)
            fprintf(pMailFile, ",%s", pMSD->ppszCC[ii]);

        fprintf(pMailFile, "\n");
    }

    fprintf(pMailFile, "MIME-Version: 1.0\n");
    fprintf(pMailFile, "X-Mailer: %s\n", VERSION_STRING);

    if (pMSD->pszAuthType != NULL)
        fprintf(pMailFile, "X-Auth: %s\n", pMSD->pszAuthType);


    char            szDateTime[256] = "";

    USmtpGetTimeStr(szDateTime, sizeof(szDateTime) - 1);

    fprintf(pMailFile, "Date: %s\n", szDateTime);


    if ((pMSD->ulFlags & SMF_HTML_BODY) ||
            ((pMSD->ppszAttach != NULL) && (pMSD->ppszAttach[0] != NULL)))
    {
        char const     *pszMimeType = (pMSD->ulFlags & SMF_HTML_BODY) ? "text/html" : "text/plain";

        fprintf(pMailFile, "Content-Type: multipart/mixed;\n"
                "\tboundary=\"%s\"\n\n", pMSD->pszBoundary);

        fprintf(pMailFile, "This is a multi-part message in MIME format.\n"
                "\n"
                "--%s\n"
                "Content-Type: %s;\n"
                "\tcharset=\"iso-8859-1\"\n"
                "Content-Transfer-Encoding: 7bit\n\n", pMSD->pszBoundary, pszMimeType);
    }
    else
        fprintf(pMailFile, "\n");


    FILE           *pBodyFile = fopen(pMSD->pszBodyFile, "rt");

    if (pBodyFile == NULL)
    {
        fclose(pMailFile);
        ErrSetErrorCode(ERR_FILE_OPEN, pMSD->pszBodyFile);
        return (ERR_FILE_OPEN);
    }

    char            szBuffer[2048];

    while (fgets(szBuffer, sizeof(szBuffer) - 1, pBodyFile) != NULL)
        fputs(szBuffer, pMailFile);

    fclose(pBodyFile);


    if ((pMSD->ppszAttach != NULL) && (pMSD->ppszAttach[0] != NULL))
    {
        for (ii = 0; pMSD->ppszAttach[ii] != NULL; ii++)
            file_encode64(pMSD->pszBoundary, pMSD->ppszAttach[ii], pMailFile);

        fprintf(pMailFile, "\n--%s--\n", pMSD->pszBoundary);
    }

    fclose(pMailFile);

    return (0);

}



static int      USmtpGetTempFile(char *pszTempFile)
{

    char            szTempPath[MAX_PATH] = "";

    GetTempPath(sizeof(szTempPath), szTempPath);

    GetTempFileName(szTempPath, "sndm", 0, pszTempFile);

    return (0);

}




static int      USmtpSendMail(BSOCK_HANDLE hBSock, MailSendData const * pMSD)
{

    char            szTempFile[MAX_PATH] = "";

    USmtpGetTempFile(szTempFile);

    if (USmtpFilePrepare(szTempFile, pMSD) < 0)
    {
        remove(szTempFile);
        return (ErrGetErrorCode());
    }

///////////////////////////////////////////////////////////////////////////////
//  Send MAIL FROM: and read result
///////////////////////////////////////////////////////////////////////////////
    int             iSvrReponse = -1;
    char            szRTXBuffer[2048] = "";

    sprintf(szRTXBuffer, "MAIL FROM:<%s>", pMSD->pszFrom);

    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szRTXBuffer, szRTXBuffer,
                            sizeof(szRTXBuffer)), 200))
    {
        remove(szTempFile);
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
        return (ErrGetErrorCode());
    }


    int             ii;

    for (ii = 0; pMSD->ppszRcpt[ii] != NULL; ii++)
    {
///////////////////////////////////////////////////////////////////////////////
//  Send RCPT TO: and read result
///////////////////////////////////////////////////////////////////////////////
        sprintf(szRTXBuffer, "RCPT TO:<%s>", pMSD->ppszRcpt[ii]);

        if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szRTXBuffer, szRTXBuffer,
                                sizeof(szRTXBuffer)), 200))
        {
            remove(szTempFile);
            if (iSvrReponse > 0)
                ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
            return (ErrGetErrorCode());
        }
    }

    if (pMSD->ppszCC != NULL)
    {
        for (ii = 0; pMSD->ppszCC[ii] != NULL; ii++)
        {
///////////////////////////////////////////////////////////////////////////////
//  Send RCPT TO: and read result
///////////////////////////////////////////////////////////////////////////////
            sprintf(szRTXBuffer, "RCPT TO:<%s>", pMSD->ppszCC[ii]);

            if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, szRTXBuffer, szRTXBuffer,
                                    sizeof(szRTXBuffer)), 200))
            {
                remove(szTempFile);
                if (iSvrReponse > 0)
                    ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
                return (ErrGetErrorCode());
            }
        }
    }

///////////////////////////////////////////////////////////////////////////////
//  Send DATA and read the "ready to receive"
///////////////////////////////////////////////////////////////////////////////
    if (!USmtpResponseClass(iSvrReponse = USmtpSendCommand(hBSock, "DATA", szRTXBuffer,
                            sizeof(szRTXBuffer)), 300))
    {
        remove(szTempFile);
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
        return (ErrGetErrorCode());
    }

///////////////////////////////////////////////////////////////////////////////
//  Send file
///////////////////////////////////////////////////////////////////////////////
    if (USmtpSendFile(szTempFile, hBSock, 1) < 0)
    {
        remove(szTempFile);
        return (ErrGetErrorCode());
    }

    remove(szTempFile);

///////////////////////////////////////////////////////////////////////////////
//  Send END OF DATA and read transfer result
///////////////////////////////////////////////////////////////////////////////
    if (BSckSendString(hBSock, ".", STD_SMTP_TIMEOUT) <= 0)
        return (ErrGetErrorCode());

    if (!USmtpResponseClass(iSvrReponse = USmtpGetResponse(hBSock, szRTXBuffer,
                            sizeof(szRTXBuffer)), 200))
    {
        if (iSvrReponse > 0)
            ErrSetErrorCode(ERR_BAD_SERVER_RESPONSE, szRTXBuffer);
        return (ErrGetErrorCode());
    }

    return (0);

}



static int      USmtpCheckMSD(MailSendData const * pMSD)
{

    if (pMSD->uSize != sizeof(MailSendData))
    {
        ErrSetErrorCode(ERR_INVALID_MSD_VERSION);
        return (ERR_INVALID_MSD_VERSION);
    }

    if (pMSD->pszBodyFile == NULL)
    {
        ErrSetErrorCode(ERR_MSD_NO_BODYFILE);
        return (ERR_MSD_NO_BODYFILE);
    }

    if (pMSD->pszSubject == NULL)
    {
        ErrSetErrorCode(ERR_MSD_NO_SUBJECT);
        return (ERR_MSD_NO_SUBJECT);
    }

    if (pMSD->pszFrom == NULL)
    {
        ErrSetErrorCode(ERR_MSD_NO_FROM);
        return (ERR_MSD_NO_FROM);
    }

    if (pMSD->ppszRcpt == NULL)
    {
        ErrSetErrorCode(ERR_MSD_NO_RCPT);
        return (ERR_MSD_NO_RCPT);
    }

    int             iRcptCount = 0;

    for (; pMSD->ppszRcpt[iRcptCount] != NULL; iRcptCount++);

    if (iRcptCount == 0)
    {
        ErrSetErrorCode(ERR_MSD_NO_RCPT);
        return (ERR_MSD_NO_RCPT);
    }



    return (0);

}



int WINAPI      USmtpLibOpen(void)
{

    if (iLibOpen == 0)
    {
        char            szSockDesc[256] = "";

        if (!SvrInitSockets(szSockDesc))
        {
            ErrSetErrorCode(ERR_NO_SOCK_SUPPORT);
            return (ERR_NO_SOCK_SUPPORT);
        }
    }

    ++iLibOpen;

    return (0);

}




int WINAPI      USmtpLibClose(void)
{

    --iLibOpen;

    if (iLibOpen == 0)
    {

        SvrCleanupSockets();

        ErrFreeErrorsInfo();
    }

    return (0);

}




int WINAPI      USmtpSendMail(char const * pszServer, int iPortNo, char const * pszDomain,
                        MailSendData const * pMSD)
{

    if (USmtpCheckMSD(pMSD) < 0)
        return (ErrGetErrorCode());


    BSOCK_HANDLE    hBSock = USmtpCreateChannel(pszServer, iPortNo, pszDomain, pMSD);

    if (hBSock == INVALID_BSOCK_HANDLE)
        return (ErrGetErrorCode());


    int             iResultCode = USmtpSendMail(hBSock, pMSD);


    USmtpCloseChannel(hBSock);

    return (iResultCode);

}




int WINAPI      USmptGetErrorInfo(int iErrorCode, char *pszBuffer, int iBufferSize)
{

    ErrorInfo      *pEI = ErrGetErrorInfo(iErrorCode);

    if (pEI == NULL)
    {
        ErrSetErrorCode(ERR_INVALID_ERROR_CODE);
        return (ERR_INVALID_ERROR_CODE);
    }

    char            szError[4096];

    sprintf(szError,
            "Error code    = %d\n"
            "Error message = \"%s\"\n"
            "Error info    = \"%s\"\n", iErrorCode, pEI->pszError,
            (pEI->pszInfo != NULL) ? pEI->pszInfo : "");

    strncpy(pszBuffer, szError, iBufferSize);
    pszBuffer[iBufferSize - 1] = '\0';

    return (0);

}





int WINAPI      USmtpArgsSendMail(int iArgCount, char *pszArgs[])
{

    int             iPort = 25,
                    iRcptCount = 0,
                    iCcCount = 0,
                    iTagsCount = 0,
                    iAttachCount = 0;
    char           *pszDomain = pszArgs[0],
                   *pszServer = "localhost",
                   *pszRcpt[MAX_RCPT],
                   *pszCC[MAX_CC],
                   *pszTags[MAX_TAGS],
                   *pszAttach[MAX_ATTACH];
    MailSendData    MSD;

    ZeroData(MSD);
    MSD.uSize = sizeof(MSD);
    MSD.ppszRcpt = pszRcpt;
    MSD.ppszCC = pszCC;
    MSD.ppszTags = pszTags;
    MSD.ppszAttach = pszAttach;
    MSD.pszBoundary = STD_BOUNDARY_DELIMITER;
    MSD.ulFlags = 0;

    for (int ii = 1; ii < iArgCount; ii++)
    {
        if (pszArgs[ii][0] == '-')
        {
            switch (pszArgs[ii][1])
            {
                case ('D'):
                    if (++ii < iArgCount)
                        pszDomain = pszArgs[ii];
                    break;

                case ('X'):
                    if (++ii < iArgCount)
                        pszServer = pszArgs[ii];
                    break;

                case ('P'):
                    if (++ii < iArgCount)
                        iPort = atoi(pszArgs[ii]);
                    break;

                case ('b'):
                    if (++ii < iArgCount)
                        MSD.pszBodyFile = pszArgs[ii];
                    break;

                case ('s'):
                    if (++ii < iArgCount)
                        MSD.pszSubject = pszArgs[ii];
                    break;

                case ('f'):
                    if (++ii < iArgCount)
                        MSD.pszFrom = pszArgs[ii];
                    break;

                case ('F'):
                    if (++ii < iArgCount)
                        MSD.pszFromExt = pszArgs[ii];
                    break;

                case ('R'):
                    if (++ii < iArgCount)
                        MSD.pszReply = pszArgs[ii];
                    break;

                case ('E'):
                    if (++ii < iArgCount)
                        MSD.pszErrors = pszArgs[ii];
                    break;

                case ('S'):
                    if (++ii < iArgCount)
                        MSD.pszSender = pszArgs[ii];
                    break;

                case ('r'):
                    if (++ii < iArgCount)
                    {
                        if (iRcptCount < (MAX_RCPT - 1))
                            pszRcpt[iRcptCount++] = pszArgs[ii];
                    }
                    break;

                case ('c'):
                    if (++ii < iArgCount)
                    {
                        if (iCcCount < (MAX_CC - 1))
                            pszCC[iCcCount++] = pszArgs[ii];
                    }
                    break;

                case ('a'):
                    if (++ii < iArgCount)
                    {
                        if (iAttachCount < (MAX_ATTACH - 1))
                            pszAttach[iAttachCount++] = pszArgs[ii];
                    }
                    break;

                case ('t'):
                    if (++ii < iArgCount)
                    {
                        if (iTagsCount < (MAX_TAGS - 1))
                            pszTags[iTagsCount++] = pszArgs[ii];
                    }
                    break;

                case ('H'):
                    MSD.ulFlags |= SMF_HTML_BODY;
                    break;

                case ('h'):
                    if (++ii < iArgCount)
                        MSD.pszAuthType = pszArgs[ii];
                    break;

                case ('u'):
                    if (++ii < iArgCount)
                        MSD.pszUsername = pszArgs[ii];
                    break;

                case ('p'):
                    if (++ii < iArgCount)
                        MSD.pszPassword = pszArgs[ii];
                    break;

                default:
                    ErrSetErrorCode(ERR_UNKNOWN_SWITCH, pszArgs[ii]);
                    return (ERR_UNKNOWN_SWITCH);
            }
        }
    }


    pszRcpt[iRcptCount] = NULL;
    pszCC[iCcCount] = NULL;
    pszTags[iTagsCount] = NULL;
    pszAttach[iAttachCount] = NULL;


    if (MSD.pszBodyFile == NULL)
    {
        ErrSetErrorCode(ERR_MSD_NO_BODYFILE);
        return (ERR_MSD_NO_BODYFILE);
    }

    if (MSD.pszSubject == NULL)
    {
        ErrSetErrorCode(ERR_MSD_NO_SUBJECT);
        return (ERR_MSD_NO_SUBJECT);
    }

    if (MSD.pszFrom == NULL)
    {
        ErrSetErrorCode(ERR_MSD_NO_FROM);
        return (ERR_MSD_NO_FROM);
    }

    if (iRcptCount == 0)
    {
        ErrSetErrorCode(ERR_MSD_NO_RCPT);
        return (ERR_MSD_NO_RCPT);
    }


    if (USmtpSendMail(pszServer, iPort, pszDomain, &MSD) < 0)
        return (ErrGetErrorCode());


    return (0);

}



int WINAPI      USmtpCmdLineSendMail(char const * pszCmdLine)
{

    int             iArgsCount = 0;
    char          **ppszArgs = USmtpGetArgs(pszCmdLine, iArgsCount);

    if (ppszArgs == NULL)
        return (ErrGetErrorCode());


    int             iResultCode = USmtpArgsSendMail(iArgsCount, ppszArgs);


    USmtpFreeArgs(ppszArgs);

    return (iResultCode);

}



BOOL WINAPI     DllMain(HINSTANCE hInstance, DWORD dwReason, LPVOID lpParam)
{

    BOOL            bLoadResult = TRUE;

    switch (dwReason)
    {
        case (DLL_PROCESS_ATTACH):
            break;

        case (DLL_PROCESS_DETACH):
            break;

        case (DLL_THREAD_ATTACH):
            break;

        case (DLL_THREAD_DETACH):
            break;
    }

    return (bLoadResult);

}



int             main(int iArgCount, char *pszArgs[])
{

    if (iArgCount == 1)
    {
        fprintf(stderr, "Use : %s  -bsfr  [-DXPFcERSa]\n"
                "-b file            Set mail body filename\n"
                "-s subject         Set mail Subject\n"
                "-f from            Set mail From\n"
                "-r rcpt            Add mail Rcpt\n"
                "-H                 Treat body file as html\n"
                "-D domain          Set HELO domain { %s }\n"
                "-X server          Set SMTP server { localhost }\n"
                "-P port            Set SMTP port { 25 }\n"
                "-F formext         Set extended From { from }\n"
                "-R reply           Set Reply-To { void }\n"
                "-S sender          Set Sender { void }\n"
                "-c cc              Add Cc { void }\n"
                "-t tag             Add mail tag ( Ie. \"TagName: TagValue\" )\n"
                "-a atchfile        Add attach file { void }\n"
                "-h auth            Set auth type ( PLAIN LOGIN CRAM-MD5 ) { void }\n"
                "-u user            Set user name for auth { void }\n"
                "-p password        Set password ( or CRAM-MD5 secret ) for auth { void }\n",
                pszArgs[0], pszArgs[0]);
        return (99);
    }

    if (USmtpLibOpen() < 0)
        return (-ErrGetErrorCode());


    int             iResultCode = USmtpArgsSendMail(iArgCount, pszArgs);


    if (iResultCode != 0)
    {
        char            szError[2048] = "";

        if (USmptGetErrorInfo(iResultCode, szError, sizeof(szError)) == 0)
            fprintf(stderr, "%s\n", szError);
        else
            fprintf(stderr, "%s\n", ErrGetErrorString(ErrGetErrorCode()));
    }

    USmtpLibClose();

    return (-iResultCode);

}
